<?php
// subscriber_route.php
header('Content-Type: application/json');

require_once __DIR__ . '/../helpers/dbConnect.php';
require_once __DIR__ . '/../pages/subscriber.php'; // Subscriber class

$db = new Database();
$subscriber = new Subscriber($db);

$method = $_SERVER['REQUEST_METHOD'];
$input = json_decode(file_get_contents('php://input'), true);
$id = $_GET['id'] ?? null;

switch ($method) {
    case 'GET':
        if ($id) {
            $data = $subscriber->read($id);
            if ($data) {
                echo json_encode($data);
            } else {
                http_response_code(404);
                echo json_encode(['error' => 'Subscriber not found']);
            }
        } else {
            // Get all subscribers
            $data = $subscriber->getAll();
            echo json_encode($data);
        }
        break;

    case 'POST':
        if (isset($input['action'])) {
            switch ($input['action']) {
                case 'update':
                    if (!isset($input['id'], $input['name'], $input['email'])) {
                        http_response_code(400);
                        echo json_encode(['error' => 'Missing id, name, or email']);
                        break;
                    }
                    $success = $subscriber->update($input['id'], $input['name'], $input['email']);
                    if ($success) {
                        echo json_encode(['message' => 'Subscriber updated']);
                    } else {
                        http_response_code(404);
                        echo json_encode(['error' => 'Subscriber not found or no change']);
                    }
                    break;

                case 'toggleBan':
                    if (!isset($input['id'])) {
                        http_response_code(400);
                        echo json_encode(['error' => 'Missing id for toggleBan']);
                        break;
                    }
                    $result = $subscriber->toggleBan($input['id']);
                    if ($result) {
                        echo json_encode([
                            'message' => 'Ban status toggled',
                            'id' => $result['subscriber_id'],
                            'new_ban_status' => $result['new_ban_status']
                        ]);
                    } else {
                        error_log("Subscriber not found or update failed for ID: " . $input['id']);
                        http_response_code(404);
                        echo json_encode(['error' => 'Subscriber not found']);
                    }
                    break;
                    case 'register_device_token':
                    if (!isset($input['subscriberId'], $input['fcmToken'])) {
                        http_response_code(400);
                        echo json_encode(['error' => 'Missing subscriberId or fcmToken']);
                        break;
                    }

                    // Optional: Get device type, default to 'android'
                    $deviceType = $input['deviceType'] ?? 'android';
                    
                    $success = $subscriber->registerDeviceToken($input['subscriberId'], $input['fcmToken'], $deviceType);

                    if ($success) {
                        http_response_code(200);
                        echo json_encode(['message' => 'Device token registered successfully']);
                    } else {
                        http_response_code(500);
                        echo json_encode(['error' => 'Failed to register device token']);
                    }
                    break;

                // NEW/MODIFIED ACTION: Checks if a subscriber exists by email and returns their data
                case 'check_subscriber_by_email':
                    if (!isset($input['email'])) {
                        http_response_code(400);
                        echo json_encode(['error' => 'Missing email']);
                        break;
                    }
                    // Fetch user data using the readByEmail method
                    $data = $subscriber->readByEmail($input['email']);

                    // Handle banned user (readByEmail returns an array with 'error' and 'http_code' 403)
                    if (isset($data['error']) && ($data['http_code'] ?? null) === 403) {
                        http_response_code(403);
                        echo json_encode(['error' => $data['error']]);
                        break;
                    }

                    if ($data) {
                        // User exists and is not banned. Successful lookup.
                        http_response_code(200);
                        echo json_encode([
                            'message' => 'Subscriber found',
                            'subscriber_id' => $data['id'],
                            'username' => $data['name'],
                            'email' => $data['email']
                        ]);
                    } else {
                        // User not found
                        http_response_code(404);
                        echo json_encode(['error' => 'Subscriber not found']);
                    }
                    break;

                default:
                    http_response_code(400);
                    echo json_encode(['error' => 'Invalid action']);
                    break;
            }
        } elseif (isset($input['name'], $input['email'])) {
            // Main subscription/creation route
            $result = $subscriber->create($input['name'], $input['email']);

            if (isset($result['error'])) {
                // Handle error: e.g., banned user
                $statusCode = ($result['error'] === 'Subscriber is banned and cannot be created.') ? 403 : 400;
                http_response_code($statusCode);
                echo json_encode(['error' => $result['error']]);
            } else {
                // Success - Check if the user was new or existing
                $is_new = ($result['status'] ?? 'new') === 'new';
                $statusCode = $is_new ? 201 : 200; // 201 for NEW, 200 for EXISTING (as requested)

                http_response_code($statusCode);
                echo json_encode([
                    'message' => $is_new ? 'Subscriber created' : 'Subscriber already exists',
                    'id' => [
                        'subscriber_id' => $result['id'],
                        'name' => $result['name'],
                        'email' => $result['email']
                    ]
                ]);
            }
        } else {
            http_response_code(400);
            echo json_encode(['error' => 'Missing required fields']);
        }
        break;

    case 'DELETE':
        if (!$id) {
            http_response_code(400);
            echo json_encode(['error' => 'Missing subscriber ID']);
            break;
        }
        $success = $subscriber->delete($id);
        if ($success) {
            echo json_encode(['message' => 'Subscriber deleted']);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Subscriber not found']);
        }
        break;

    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        break;
}
?>