<?php
class Subscriber {
    private $db;

    public function __construct(Database $database) {
        $this->db = $database;
    }
    private function generateUUID(): string {
        return sprintf(
            '%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }

    public function create($name, $email) {
        // Check if the subscriber already exists
        $sql = "SELECT * FROM subscribers WHERE email = :email";
        $existingSubscriber = $this->db->runQuery($sql, ['email' => $email]);

        if ($existingSubscriber) {
            $subscriber = $existingSubscriber[0];

            // If the subscriber exists and is banned
            if ($subscriber['banned'] == 1) {
                return [
                    'error' => 'Subscriber is banned and cannot be created.',
                    'http_code' => 403
                ];
            }

            // Subscriber exists and is NOT banned. Return existing data.
            return [
                'id' => $subscriber['id'],
                'name' => $subscriber['name'],
                'email' => $subscriber['email'],
                'status' => 'existing' // Indicates successful login/retrieval for existing user
            ];
        }

        // Proceed with NEW user creation
        $subscriber_id = $this->generateUUID();
        $sql = "INSERT INTO subscribers (id, name, email) VALUES (:subscriber_id, :name, :email)";
        $result = $this->db->runQuery($sql, [
            'subscriber_id' => $subscriber_id,
            'name' => $name,
            'email' => $email
        ]);

        if ($result !== false) {
            // New user created
            return [
                'id' => $subscriber_id,
                'name' => $name,
                'email' => $email,
                'status' => 'new' // Indicates a new user was created
            ];
        } else {
            return [
                'error' => 'Database insert failed.'
            ];
        }
    }

    // METHOD: Used by 'check_subscriber_by_email' action
    public function readByEmail($email) {
        $sql = "SELECT * FROM subscribers WHERE email = :email";
        $result = $this->db->runQuery($sql, ['email' => $email]);
        if ($result) {
            $subscriber = $result[0];
            if ($subscriber['banned'] == 1) {
                return ['error' => 'User is banned.', 'http_code' => 403];
            }
            return $subscriber;
        }
        return false;
    }

    public function getAll() {
        $sql = "SELECT * FROM subscribers";
        return $this->db->runQuery($sql);
    }


    public function read($id) {
        $sql = "SELECT * FROM subscribers WHERE id = :id";
        $result = $this->db->runQuery($sql, ['id' => $id]);
        return $result[0] ?? null;
    }

    public function update($id, $name, $email) {
        $sql = "UPDATE subscribers SET name = :name, email = :email WHERE id = :id";
        return $this->db->runQuery($sql, compact('id', 'name', 'email'));
    }

    public function delete($id) {
        $sql = "DELETE FROM subscribers WHERE id = :id";
        return $this->db->runQuery($sql, ['id' => $id]);
    }

    public function toggleBan($id) {
        try {
            $sql = "SELECT banned FROM subscribers WHERE id = :id";
            $stmt = $this->db->prepare($sql);
            $stmt->execute(['id' => $id]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($result) {
                $currentBanStatus = $result['banned'];
                $newBanStatus = ($currentBanStatus == 0) ? 1 : 0;

                $updateSql = "UPDATE subscribers SET banned = :banned WHERE id = :id";
                $updateStmt = $this->db->prepare($updateSql);
                $updateResult = $updateStmt->execute([
                    'banned' => $newBanStatus,
                    'id' => $id
                ]);

                if ($updateResult) {
                    return [
                        'subscriber_id' => $id,
                        'new_ban_status' => $newBanStatus
                    ];
                } else {
                    error_log("Failed to update ban status for subscriber ID $id");
                    return false;
                }
            } else {
                return false;
            }
        } catch (PDOException $e) {
            error_log("Database error in toggleBan: " . $e->getMessage());
            return false;
        }
    }


}
?>