<?php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, GET, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");

if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit;
}

require __DIR__.'/../vendor/autoload.php';
require_once __DIR__ . '/../helpers/dbConnect.php';

use Kreait\Firebase\Factory;
use Kreait\Firebase\Messaging\CloudMessage;
use Kreait\Firebase\Messaging\Notification;

header('Content-Type: application/json');

$db = new Database();
$method = $_SERVER['REQUEST_METHOD'];

// Input handling for both JSON and Form Data
$input = json_decode(file_get_contents('php://input'), true) ?? $_POST;

switch ($method) {

    /**
     * GET: Fetch Notifications + "Checked" Status
     * Requires: ?subscriber_id=XYZ
     */
  case 'GET':
        $subscriber_id = $_GET['subscriber_id'] ?? null;
        if (!$subscriber_id) {
            http_response_code(400);
            echo json_encode(['error' => 'Subscriber ID required.']);
            break;
        }

        try {
            // Fetch ALL notifications and check read status for this user
            $sql = "SELECT n.*, 
                    IF(r.subscriber_id IS NOT NULL, 1, 0) as is_read
                    FROM app_notifications n
                    LEFT JOIN notification_reads r ON n.notification_id = r.notification_id 
                         AND r.subscriber_id = :sid
                    WHERE n.created_at >= NOW() - INTERVAL 48 HOUR 
                    ORDER BY n.created_at DESC";
            
            $notifications = $db->runQuery($sql, ['sid' => $subscriber_id]);
            $notifications = $notifications ?: [];

            // Calculate unread count from the results
            $unreadCount = 0;
            foreach ($notifications as $notif) {
                if ($notif['is_read'] == 0) {
                    $unreadCount++;
                }
            }

            echo json_encode([
                'unread_count' => $unreadCount,
                'notifications' => $notifications
            ]);

        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => 'Fetch Error: ' . $e->getMessage()]);
        }
        break;

    /**
     * POST: Send Notification OR Mark as Checked
     */
    case 'POST':
        $action = $_GET['action'] ?? 'send';

        if ($action === 'track_click') {
            $nid = $input['notification_id'] ?? null;
            $sid = $input['subscriber_id'] ?? null;
            if ($nid && $sid) {
                $db->runQuery("UPDATE app_notifications SET click_count = click_count + 1 WHERE notification_id = :nid", ['nid' => $nid]);
                $db->runQuery("INSERT IGNORE INTO notification_reads (notification_id, subscriber_id) VALUES (:nid, :sid)", ['nid' => $nid, 'sid' => $sid]);
                echo json_encode(['success' => true]);
            } 
        }
        else {
            $title = $_POST['title'] ?? null;
            $body = $_POST['body'] ?? null;

            if (!$title || !$body) {
                http_response_code(400);
                echo json_encode(['error' => 'Missing title or body.']);
                exit;
            }

            // --- DEBUG: LOG INCOMING DATA ---
            $debugData = [
                'time' => date('Y-m-d H:i:s'),
                'title' => $title,
                'body' => $body,
                'files' => $_FILES
            ];
            file_put_contents('debug_log.txt', "--- NEW ATTEMPT ---\n" . print_r($debugData, true), FILE_APPEND);

            $imageUrl = null;
            if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
                $uploadDir = __DIR__ . '/../upload/';
                if (!is_dir($uploadDir)) mkdir($uploadDir, 0777, true);
                $filename = time() . '_' . basename($_FILES['image']['name']);
                if (move_uploaded_file($_FILES['image']['tmp_name'], $uploadDir . $filename)) {
                    $imageUrl = 'https://api.237showbiz.com/api/upload/' . $filename;
                }
            }

            $notificationId = uniqid('notif_');

            // --- REVISED DATABASE SAVE WITH ERROR CAPTURE ---
            try {
                // Ensure click_count starts at 0 and created_at uses current time
                $sql = "INSERT INTO app_notifications (notification_id, title, body, image_url, click_count, created_at) 
                        VALUES (:nid, :title, :body, :img, 0, NOW())";
                
                $params = [
                    'nid'   => $notificationId,
                    'title' => $title,
                    'body'  => $body,
                    'img'   => $imageUrl
                ];

                $result = $db->runQuery($sql, $params);

                // Check if runQuery returned false (means database failed to execute)
                if ($result === false) {
                    throw new \Exception("Database execution failed. The query returned false.");
                }

                file_put_contents('debug_log.txt', "DB Save Success: ID $notificationId\n", FILE_APPEND);

            } catch (\Exception $e) {
                // Return full error details to Flutter
                http_response_code(500);
                $errorResponse = [
                    'success' => false,
                    'error' => 'DB Save Failed: ' . $e->getMessage(),
                    'sql_attempted' => $sql,
                    'params_sent' => $params
                ];
                file_put_contents('debug_log.txt', "DB ERROR: " . $e->getMessage() . "\n", FILE_APPEND);
                echo json_encode($errorResponse);
                exit;
            }

            // --- FCM SENDING LOGIC ---
            try {
                $projectRoot = dirname(__DIR__);
                $serviceAccountPath = $projectRoot . '/upload/showbizapp-f8f9d-firebase-adminsdk-fbsvc-1f2dcd4e75.json';
                
                if (!file_exists($serviceAccountPath)) {
                    throw new \Exception("Firebase JSON file not found at $serviceAccountPath");
                }

                $messaging = (new Factory)->withServiceAccount($serviceAccountPath)->createMessaging();
                $notif = Notification::create($title, $body);
                if ($imageUrl) $notif = $notif->withImageUrl($imageUrl);

                $message = CloudMessage::withTarget('topic', 'all')
                    ->withNotification($notif)
                    ->withData(['notification_id' => $notificationId]);

                $messaging->send($message);
                
                file_put_contents('debug_log.txt', "FCM Sent Successfully\n", FILE_APPEND);
                echo json_encode(['success' => true, 'id' => $notificationId]);

            } catch (\Exception $e) {
                file_put_contents('debug_log.txt', "FCM ERROR: " . $e->getMessage() . "\n", FILE_APPEND);
                echo json_encode(['success' => true, 'id' => $notificationId, 'fcm_warning' => $e->getMessage()]);
            }
        }
        break;

    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        break;
}