<?php
class Subscriber {
    private $db;

    public function __construct(Database $database) {
        $this->db = $database;
    }
        private function generateUUID(): string {
        return sprintf(
            '%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }

    public function create($name, $email) {
        // Check if the subscriber already exists
        $sql = "SELECT * FROM subscribers WHERE email = :email";
        $existingSubscriber = $this->db->runQuery($sql, ['email' => $email]);

        if ($existingSubscriber) {
            $subscriber = $existingSubscriber[0];

            // If the subscriber exists and is banned
            if ($subscriber['banned'] == 1) {
                return [
                    'error' => 'Subscriber is banned and cannot be created.',
                    'http_code' => 403
                ];
            }

            // Return existing subscriber details as if account was created
            return [
                'id' => $subscriber['id'],
                'name' => $subscriber['name'],
                'email' => $subscriber['email']
            ];
        }

        // Generate a new UUID for the subscriber
        $subscriber_id = $this->generateUUID();

        // Prepare the SQL query to create a new subscriber
        $sql = "INSERT INTO subscribers (id, name, email) VALUES (:subscriber_id, :name, :email)";
        $result = $this->db->runQuery($sql, [
            'subscriber_id' => $subscriber_id,
            'name' => $name,
            'email' => $email
        ]);

        // Check if the query was successful
        if ($result !== false) {
            return [
                'id' => $subscriber_id,
                'name' => $name,
                'email' => $email
            ];
        } else {
            return [
                'error' => 'Database insert failed.'
            ];
        }
    }
    
     /**
     * NEW METHOD: Register a device token for a subscriber.
     * It prevents duplicate tokens for the same user.
     *
     * @param string $subscriber_id The UUID of the subscriber.
     * @param string $fcm_token The FCM token from the device.
     * @param string $device_type The type of device (e.g., 'android', 'ios').
     * @return bool True on success, false on failure.
     */
    public function registerDeviceToken($subscriber_id, $fcm_token, $device_type = 'android') {
        // First, check if this exact token already exists for this subscriber to avoid duplicates
        $checkSql = "SELECT id FROM device_tokens WHERE subscriber_id = :subscriber_id AND fcm_token = :fcm_token";
        $existingToken = $this->db->runQuery($checkSql, [
            'subscriber_id' => $subscriber_id,
            'fcm_token' => $fcm_token
        ]);

        // If the token already exists, do nothing and return success.
        if ($existingToken) {
            return true;
        }

        // If the token is new for this user, insert it.
        $insertSql = "INSERT INTO device_tokens (subscriber_id, fcm_token, device_type) VALUES (:subscriber_id, :fcm_token, :device_type)";
        $result = $this->db->runQuery($insertSql, [
            'subscriber_id' => $subscriber_id,
            'fcm_token' => $fcm_token,
            'device_type' => $device_type
        ]);

        return $result !== false;
    }


 public function getAll() {
        $sql = "SELECT * FROM subscribers";
        return $this->db->runQuery($sql);
    }



    public function read($id) {
        $sql = "SELECT * FROM subscribers WHERE id = :id";
        $result = $this->db->runQuery($sql, ['id' => $id]);
        return $result[0] ?? null;
    }

    public function update($id, $name, $email) {
        $sql = "UPDATE subscribers SET name = :name, email = :email WHERE id = :id";
        return $this->db->runQuery($sql, compact('id', 'name', 'email'));
    }

    public function delete($id) {
        $sql = "DELETE FROM subscribers WHERE id = :id";
        return $this->db->runQuery($sql, ['id' => $id]);
    }

  public function toggleBan($id) {
    try {
        $sql = "SELECT banned FROM subscribers WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        $stmt->execute(['id' => $id]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($result) {
            $currentBanStatus = $result['banned'];
            $newBanStatus = ($currentBanStatus == 0) ? 1 : 0;

            $updateSql = "UPDATE subscribers SET banned = :banned WHERE id = :id";
            $updateStmt = $this->db->prepare($updateSql);
            $updateResult = $updateStmt->execute([
                'banned' => $newBanStatus,
                'id' => $id
            ]);

            if ($updateResult) {
                return [
                    'subscriber_id' => $id,
                    'new_ban_status' => $newBanStatus
                ];
            } else {
                error_log("Failed to update ban status for subscriber ID $id");
                return false;
            }
        } else {
            return false; // Subscriber not found
        }
    } catch (PDOException $e) {
        error_log("Database error in toggleBan: " . $e->getMessage());
        return false;
    }
}


}
