<?php

class Comment {
    private $db;

    public function __construct(Database $database) {
        $this->db = $database;
    }

    // UUID Generator
    private function generateUUID(): string {
        return sprintf(
            '%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }

    // NEW: Helper function to convert DB format to ISO 8601 UTC for the API
    private function formatTimestampForApi($timestamp): string {
        // Assume the stored 'Y-m-d H:i:s' is UTC, and append 'T' and 'Z'
        if (empty($timestamp) || $timestamp === 'null') return '';
        // Replace space with 'T' and append 'Z' (Zulu time) for full ISO 8601 UTC format.
        return str_replace(' ', 'T', $timestamp) . 'Z';
    }

    // --- SPAM PREVENTION METHODS ---

    private function checkDuplicateContent($text, $subscriber_id, $post_id): bool {
        $sql = "SELECT text FROM comments
                WHERE subscriber_id = :subscriber_id AND post_id = :post_id
                ORDER BY timestamp DESC LIMIT 1";
        $stmt = $this->db->prepare($sql);
        $stmt->execute(['subscriber_id' => $subscriber_id, 'post_id' => $post_id]);
        $lastCommentText = $stmt->fetch(PDO::FETCH_COLUMN);

        if ($lastCommentText && trim($lastCommentText) === trim($text)) {
            return false;
        }
        return true;
    }

    private function checkRateLimit($subscriber_id, $timeWindowSeconds = 60, $maxComments = 5): bool {
        $cutoffTime = date('Y-m-d H:i:s', time() - $timeWindowSeconds);

        $sql = "SELECT COUNT(*) FROM comments
                WHERE subscriber_id = :subscriber_id AND timestamp >= :cutoffTime";
        $stmt = $this->db->prepare($sql);
        $stmt->execute(['subscriber_id' => $subscriber_id, 'cutoffTime' => $cutoffTime]);
        $commentCount = $stmt->fetchColumn();

        if ($commentCount >= $maxComments) {
            return false;
        }
        return true;
    }

    // --- END SPAM PREVENTION METHODS ---

    // Create a comment (top-level or reply)
    public function create($text, $subscriber_id, $post_id = null, $comment_count = 0, $timestamp = null, $parent_id = null) {
        try {
            // Check ban status
            $banCheckSql = "SELECT banned FROM subscribers WHERE id = :subscriber_id";
            $stmt = $this->db->prepare($banCheckSql);
            $stmt->execute(['subscriber_id' => $subscriber_id]);
            $subscriber = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$subscriber) {
                http_response_code(404);
                echo json_encode(['error' => 'Subscriber not found', "id" => (string)$subscriber_id]);
                exit;
            }

            if ($subscriber['banned'] == 1) {
                http_response_code(403);
                echo json_encode(['error' => 'Subscriber is banned']);
                exit;
            }

            // ==========================================================
            // NEW SPAM CHECKS
            // ==========================================================
            if (!$this->checkDuplicateContent($text, $subscriber_id, $post_id)) {
                http_response_code(429);
                echo json_encode(['error' => 'Please vary your comment content. Identical comment submitted recently.']);
                exit;
            }

            if (!$this->checkRateLimit($subscriber_id)) {
                http_response_code(429);
                echo json_encode(['error' => 'You are posting comments too quickly. Please wait a moment.']);
                exit;
            }

            // ==========================================================

            // Insert comment
            $comment_id = $this->generateUUID();

            // ✅ FIXED: Always store UTC timestamps to prevent timezone drift
            $utcNow = new DateTime('now', new DateTimeZone('UTC'));
            $timestamp = $timestamp ?? $utcNow->format('Y-m-d H:i:s'); // Retain format for DATETIME column

            $sql = "INSERT INTO comments (comment_id, text, subscriber_id, post_id, comment_count, timestamp, parent_id)
                    VALUES (:comment_id, :text, :subscriber_id, :post_id, :comment_count, :timestamp, :parent_id)";
            $success = $this->db->runQuery($sql, [
                'comment_id' => $comment_id,
                'text' => $text,
                'subscriber_id' => $subscriber_id,
                'post_id' => $post_id,
                'comment_count' => $comment_count,
                'timestamp' => $timestamp,
                'parent_id' => $parent_id
            ]);

            if (!$success) {
                http_response_code(500);
                echo json_encode(['error' => 'Database error', 'details' => $this->db->errorInfo()]);
                exit;
            }

            http_response_code(201);
            // MODIFIED: Return the newly created comment ID and the timestamp in the new ISO 8601 UTC format
            return [
                'comment_id' => $comment_id,
                'timestamp' => $this->formatTimestampForApi($timestamp)
            ];

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => 'Exception occurred', 'details' => $e->getMessage()]);
        }
    }

    // Update comment text and count
    public function update($comment_id, $text, $comment_count = null) {
        $sql = "UPDATE comments SET text = :text, comment_count = :comment_count WHERE comment_id = :comment_id";
        return $this->db->runQuery($sql, [
            'comment_id' => $comment_id,
            'text' => $text,
            'comment_count' => $comment_count
        ]);
    }

    // Delete a comment
    public function delete($comment_id) {
        $sql = "DELETE FROM comments WHERE comment_id = :comment_id";
        return $this->db->runQuery($sql, [
            'comment_id' => $comment_id
        ]);
    }

    // Fetch all comments for a post and build nested replies
    public function getCommentsByPost($post_id) {
        $sql = "
            SELECT
                comments.comment_id,
                comments.parent_id,
                comments.text,
                comments.subscriber_id,
                subscribers.name AS subscriber_name,
                comments.comment_count,
                comments.timestamp,
                comments.post_id,
                comments.like_count
            FROM comments
            JOIN subscribers ON comments.subscriber_id = subscribers.id
            WHERE comments.post_id = :post_id
            ORDER BY comments.timestamp ASC
        ";
        $allComments = $this->db->runQuery($sql, ['post_id' => $post_id]);

        // FIX: Format timestamps for API output before building the tree
        foreach ($allComments as &$comment) {
            $comment['timestamp'] = $this->formatTimestampForApi($comment['timestamp']);
        }

        return $this->buildCommentTree($allComments);
    }

    public function getById($comment_id) {
        $sql = "SELECT * FROM comments WHERE comment_id = :comment_id";
        $result = $this->db->runQuery($sql, ['comment_id' => $comment_id]);
        return $result ? $result[0] ?? null : null;
    }

    private function buildCommentTree(array $comments, $parentId = null) {
        $branch = [];

        foreach ($comments as $comment) {
            if ($comment['parent_id'] == $parentId) {
                $children = $this->buildCommentTree($comments, $comment['comment_id']);
                $comment['replies'] = $children;
                $branch[] = $comment;
            }
        }

        return $branch;
    }

    public function getAll() {
        $sql = "
            SELECT
                comments.comment_id,
                comments.parent_id,
                comments.text,
                comments.subscriber_id,
                subscribers.name AS subscriber_name,
                comments.comment_count,
                comments.timestamp,
                comments.post_id,
                comments.like_count
            FROM comments
            JOIN subscribers ON comments.subscriber_id = subscribers.id
            ORDER BY comments.timestamp DESC
        ";
        $results = $this->db->runQuery($sql);

        // FIX: Format timestamps for API output
        foreach ($results as &$comment) {
            $comment['timestamp'] = $this->formatTimestampForApi($comment['timestamp']);
        }

        return $results;
    }

    public function read($post_id) {
        $sql = "SELECT * FROM comments WHERE post_id = :post_id";
        $result = $this->db->runQuery($sql, ['post_id' => $post_id]);
        return $result[0] ?? null;
    }
}