<?php

class CommentInteraction {
    private $db;

    public function __construct(Database $database) {
        $this->db = $database;
    }

    private function generateUUID(): string {
        return sprintf(
            '%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }

    // Toggle like/unlike and update like count accordingly
public function toggleLike($postId, $subscriber_id, $like) {
    // 1 = Like, 0 = Unlike
    $val = $like ? 1 : 0;

    try {
        $sql = "INSERT INTO post_likes (postId, subscriber_id, like_count) 
                VALUES (:postId, :subscriber_id, :val)
                ON DUPLICATE KEY UPDATE like_count = :val_update";

        $this->db->runQuery($sql, [
            'postId'        => $postId,
            'subscriber_id' => $subscriber_id,
            'val'           => $val,
            'val_update'    => $val
        ]);

        // Return the fresh total for Postman
        return [
            'success' => true,
            'new_count' => $this->getLikeCount($postId)
        ];

    } catch (\Exception $e) {
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

public function getLikeCount($postId) {
    // This sums all the '1's in the table for this specific post
    $sql = "SELECT SUM(like_count) as total FROM post_likes WHERE postId = :id";
    $result = $this->db->runQuery($sql, ['id' => $postId]);
    
    $count = isset($result[0]['total']) ? (int)$result[0]['total'] : 0;
    
    // Safety check: ensure we never return a negative number
    return max(0, $count);
}

private function resetNegativeLike($comment_id) {
    $this->db->runQuery("UPDATE comment_likes SET like_count = 0 WHERE comment_id = :id", ['id' => $comment_id]);
}

    // Reply to a comment
    public function replyToComment($parent_comment_id, $subscriber_id, $text, $post_id) {
        $comment_id = $this->generateUUID();
        $sql = "INSERT INTO comments (comment_id, parent_id, subscriber_id, text, post_id, timestamp)
                VALUES (:comment_id, :parent_id, :subscriber_id, :text, :post_id, :timestamp)";

        $params = [
            'comment_id' => $comment_id,
            'parent_id' => $parent_comment_id,
            'subscriber_id' => $subscriber_id,
            'text' => $text,
            'post_id' => $post_id,
            'timestamp' => date('Y-m-d H:i:s')
        ];

        // This is a crucial change: add a check for the result of runQuery
        $result = $this->db->runQuery($sql, $params);

        // Assuming a successful runQuery returns a non-false value (e.g., an array)
        if ($result !== false && !isset($result['error'])) {
            return ['success' => true];
        } else {
            // Return the specific error message from the database
            return ['success' => false, 'error' => $result['error'] ?? 'Unknown database error'];
        }
    }

    // Get replies for a comment
    public function getReplies($comment_id) {
        // Corrected SQL to fetch from the `comments` table using the parent_id
        $sql = "SELECT * FROM comments WHERE parent_id = :comment_id ORDER BY timestamp ASC";
        return $this->db->runQuery($sql, ['comment_id' => $comment_id]);
    }

    /*Get like count for a comment
    public function getLikeCount($comment_id) {
        $sql = "SELECT COUNT(*) as like_count FROM comment_likes WHERE comment_id = :comment_id";
        $result = $this->db->runQuery($sql, ['comment_id' => $comment_id]);
        return $result[0]['like_count'] ?? 0;
    }*/

    public function delete($comment_id, $subscriber_id) {
        $sql = "DELETE FROM comments WHERE comment_id = :comment_id";
        $result = $this->db->runQuery($sql, ['comment_id' => $comment_id]);
        return ['success' => $result !== false];
    }
}